// pages/api/me/index.ts
import type { NextApiRequest, NextApiResponse } from "next";
import { prisma } from "@/lib/prisma";
import { requireSession } from "@/lib/auth";
import { z } from "zod";

const PatchSchema = z.object({
  name: z.string().min(1).max(100).optional(),
  email: z.string().email().optional(),
  ethAddress: z
    .string()
    .regex(/^0x[a-fA-F0-9]{40}$/, "Invalid Ethereum address")
    .optional(),
});

export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  const session = await requireSession(req, res);
  if (!session) return;

  const userId = (session.user as any).id as string;

  if (req.method === "GET") {
    const me = await prisma.user.findUnique({
      where: { id: userId },
      select: {
        id: true,
        email: true,
        name: true,
        role: true,
        createdAt: true,
        updatedAt: true,
      },
    });
    return res.status(200).json({ ok: true, user: me });
  }

  if (req.method === "PATCH") {
    const parsed = PatchSchema.safeParse(req.body);
    if (!parsed.success) {
      return res.status(400).json({
        ok: false,
        error: "Invalid input",
        details: parsed.error.flatten(),
      });
    }

    const { name, email, ethAddress } = parsed.data;

    // If changing email, enforce uniqueness
    if (email) {
      const exists = await prisma.user.findUnique({ where: { email } });
      if (exists && exists.id !== userId) {
        return res
          .status(409)
          .json({ ok: false, error: "Email already in use" });
      }
    }

    const updated = await prisma.user.update({
      where: { id: userId },
      data: {
        ...(name !== undefined ? { name } : {}),
        ...(email ? { email } : {}),
        ...(ethAddress !== undefined ? { ethAddress } : {}),
      },
      select: {
        id: true,
        email: true,
        name: true,
        role: true,
        ethAddress: true,
        updatedAt: true,
      },
    });

    return res.status(200).json({ ok: true, user: updated });
  }

  return res.status(405).json({ ok: false, error: "Method not allowed" });
}
